<?php
/**
 * API IPVA RJ - Consulta Portal Fazenda RJ
 * Combina dados do veículo + dados de pagamentos
 * 
 * Uso: api.php?renavam=01005706384
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Obter renavam da URL
$renavam = $_GET['renavam'] ?? '';

// Validar renavam
if (empty($renavam)) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Parâmetro renavam é obrigatório',
        'exemplo' => 'api.php?renavam=01005706384'
    ]);
    exit;
}

// Limpar renavam (apenas números)
$renavam = preg_replace('/[^0-9]/', '', $renavam);

if (strlen($renavam) < 9 || strlen($renavam) > 11) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Renavam inválido. Deve ter entre 9 e 11 dígitos.'
    ]);
    exit;
}

/**
 * Função para fazer requisição cURL para API da Fazenda RJ
 */
function fetchFazendaRJ(string $endpoint, string $renavam): array {
    $url = "https://darj-ipva-api.fazenda.rj.gov.br/darj-ipva-api/api/publico/darj/{$endpoint}/{$renavam}";
    
    $headers = [
        'User-Agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/143.0.0.0 Safari/537.36',
        'Accept: application/json, text/plain, */*',
        'Accept-Encoding: gzip, deflate, br',
        'Accept-Language: pt-BR,pt;q=0.9,en-US;q=0.8,en;q=0.7',
        'sec-ch-ua: "Google Chrome";v="143", "Chromium";v="143", "Not A(Brand";v="24"',
        'sec-ch-ua-mobile: ?0',
        'sec-ch-ua-platform: "Linux"',
        'Origin: https://darj-ipva-web.fazenda.rj.gov.br',
        'Referer: https://darj-ipva-web.fazenda.rj.gov.br/',
        'Sec-Fetch-Site: same-site',
        'Sec-Fetch-Mode: cors',
        'Sec-Fetch-Dest: empty',
    ];
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_ENCODING => '',
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_FOLLOWLOCATION => true,
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($error) {
        return [
            'success' => false,
            'error' => "Erro cURL: {$error}",
            'http_code' => $httpCode
        ];
    }
    
    $data = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            'success' => false,
            'error' => 'Resposta inválida da API',
            'http_code' => $httpCode,
            'raw' => substr($response, 0, 500)
        ];
    }
    
    return [
        'success' => true,
        'http_code' => $httpCode,
        'data' => $data
    ];
}

// Buscar dados do veículo
$veiculoResult = fetchFazendaRJ('dados-veiculo', $renavam);

// Buscar dados de cobrança
$cobrancaResult = fetchFazendaRJ('dados-cobranca', $renavam);

// Montar resposta combinada
$response = [
    'success' => true,
    'renavam' => $renavam,
    'timestamp' => date('c'),
    'veiculo' => null,
    'cobranca' => null,
    'errors' => []
];

// Processar resultado do veículo
if ($veiculoResult['success'] && $veiculoResult['http_code'] === 200) {
    $response['veiculo'] = $veiculoResult['data'];
} else {
    $response['errors'][] = [
        'endpoint' => 'dados-veiculo',
        'error' => $veiculoResult['error'] ?? 'Erro desconhecido',
        'http_code' => $veiculoResult['http_code'] ?? null
    ];
}

// Processar resultado da cobrança
if ($cobrancaResult['success'] && $cobrancaResult['http_code'] === 200) {
    $response['cobranca'] = $cobrancaResult['data'];
} else {
    $response['errors'][] = [
        'endpoint' => 'dados-cobranca',
        'error' => $cobrancaResult['error'] ?? 'Erro desconhecido',
        'http_code' => $cobrancaResult['http_code'] ?? null
    ];
}

// Se ambos falharam, marcar success como false
if ($response['veiculo'] === null && $response['cobranca'] === null) {
    $response['success'] = false;
    http_response_code(502);
}

// Remover array de erros se vazio
if (empty($response['errors'])) {
    unset($response['errors']);
}

// ==================== VERIFICAR SE HÁ DÉBITOS PENDENTES ====================
// Se todos os pagamentos têm data de pagamento (quitados), gerar licenciamento pendente 2026

$temDebitoPendente = false;
$anoAtual = (int)date('Y');

if (!empty($response['cobranca']) && is_array($response['cobranca'])) {
    foreach ($response['cobranca'] as $pagamento) {
        // Verificar se existe algum pagamento sem data (pendente)
        if (empty($pagamento['dtPagamento'])) {
            $temDebitoPendente = true;
            break;
        }
    }
}

// Se não há débitos pendentes, adicionar licenciamento 2026
if (!$temDebitoPendente && $response['veiculo'] !== null) {
    $response['debitos_pendentes'] = [
        [
            'id' => 'lic_' . uniqid(),
            'tipo' => 'Licenciamento',
            'exercicio' => $anoAtual,
            'descricao' => "LICENCIAMENTO {$anoAtual}",
            'valor' => 281.29,
            'valor_formatado' => 'R$ 281,29',
            'vencimento' => "31/12/{$anoAtual}",
            'vencimento_iso' => "{$anoAtual}-12-31",
            'situacao' => 'Pendente',
            'observacao' => 'Taxa de Licenciamento Anual'
        ]
    ];
    $response['total_debitos'] = 281.29;
    $response['total_debitos_formatado'] = 'R$ 281,29';
    $response['situacao_veiculo'] = 'LICENCIAMENTO PENDENTE';
} else if ($temDebitoPendente) {
    $response['situacao_veiculo'] = 'DÉBITOS PENDENTES';
} else {
    $response['situacao_veiculo'] = 'REGULAR';
}

echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
